const fs = require('fs');
const os = require('os');
const path = require('path');
const { Worker } = require('worker_threads');
const logger = require('./logger');
const { findFFmpeg, getVideoCodec, getHardwareDecoder, parseDecoderFromOutput, getDecoderType } = require('./ffmpeg');
const gpuModule = require('./visualHighlightGpu');

const RESIZE_WIDTH = 32;
const RESIZE_HEIGHT = 32;
const DEFAULT_FRAME_SKIP = 5;
const DEFAULT_BATCH_SIZE = 128;
const DEFAULT_SIMILARITY_THRESHOLD = 0.6;

function findFFmpegSync() {
  return findFFmpeg();
}

async function getVideoMetadata(videoPath) {
  return new Promise((resolve, reject) => {
    const { spawn } = require('child_process');
    const ffmpeg = findFFmpegSync();
    
    const args = [
      '-i', videoPath,
      '-hide_banner'
    ];
    
    const child = spawn(ffmpeg, args, { windowsHide: true });
    let stderrOutput = '';
    
    child.stderr.on('data', (data) => {
      stderrOutput += data.toString();
    });
    
    child.on('error', reject);
    
    child.on('close', () => {
      const widthMatch = stderrOutput.match(/(\d{2,5})x(\d{2,5})/);
      const fpsMatch = stderrOutput.match(/(\d+(?:\.\d+)?)\s*fps/);
      const durationMatch = stderrOutput.match(/Duration: (\d{2}):(\d{2}):(\d{2})\.(\d{2})/);
      
      let duration = 0;
      if (durationMatch) {
        const hours = parseInt(durationMatch[1], 10);
        const minutes = parseInt(durationMatch[2], 10);
        const seconds = parseInt(durationMatch[3], 10);
        const centiseconds = parseInt(durationMatch[4], 10);
        duration = hours * 3600 + minutes * 60 + seconds + centiseconds / 100;
      }
      
      resolve({
        width: widthMatch ? parseInt(widthMatch[1], 10) : 1920,
        height: widthMatch ? parseInt(widthMatch[2], 10) : 1080,
        fps: fpsMatch ? parseFloat(fpsMatch[1]) : 30,
        duration
      });
    });
  });
}

function scaleRegionToNative(region, targetVideoWidth, targetVideoHeight) {
  const displayWidth = region.displayWidth || 1;
  const displayHeight = region.displayHeight || 1;
  // Use TARGET video dimensions if provided, otherwise fall back to stored region dimensions
  const videoWidth = targetVideoWidth || region.videoWidth || displayWidth;
  const videoHeight = targetVideoHeight || region.videoHeight || displayHeight;
  
  const scaleX = videoWidth / displayWidth;
  const scaleY = videoHeight / displayHeight;
  
  return {
    x: Math.round(region.x * scaleX),
    y: Math.round(region.y * scaleY),
    width: Math.round(region.width * scaleX),
    height: Math.round(region.height * scaleY)
  };
}

async function extractRegionAsImage(videoPath, timestamp, region, outputPath, targetVideoWidth, targetVideoHeight) {
  let scaledRegion = region;

  if (region.displayWidth && region.displayHeight) {
    scaledRegion = scaleRegionToNative(region, targetVideoWidth, targetVideoHeight);
  }

  const { x, y, width, height } = scaledRegion;

  const codec = await getVideoCodec(videoPath);
  const hwDecoder = await getHardwareDecoder(codec);

  const { spawn } = require('child_process');
  const ffmpeg = findFFmpegSync();

  // Build FFmpeg args with explicit hardware decoder if available
  const args = [];

  if (hwDecoder) {
    args.push('-hwaccel', hwDecoder.hwaccel);
    logger.log(`[VisualHighlight] extractRegionAsImage: ${path.basename(videoPath)}`);
    logger.log(`[VisualHighlight]   Timestamp: ${timestamp}s, Codec: ${codec || 'unknown'}, Using GPU decoder: ${hwDecoder.decoder} (${hwDecoder.hwaccel})`);
  } else {
    args.push('-hwaccel', 'auto');
    logger.log(`[VisualHighlight] extractRegionAsImage: ${path.basename(videoPath)}`);
    logger.log(`[VisualHighlight]   Timestamp: ${timestamp}s, Codec: ${codec || 'unknown'}, HW accel: auto (using software decode)`);
  }

  args.push('-hide_banner', '-loglevel', 'info');
  args.push('-ss', String(timestamp), '-i', videoPath);

  // Explicitly specify the decoder (needed for hardware decode)
  if (hwDecoder) {
    args.push('-c:v', hwDecoder.decoder);
  }

  args.push(
    '-vframes', '1',
    '-vf', `crop=${width}:${height}:${x}:${y},scale=${RESIZE_WIDTH}:${RESIZE_HEIGHT}:flags=neighbor`,
    '-f', 'image2',
    '-c:v', 'png',
    '-compression_level', '0',
    outputPath
  );

  logger.log(`[VisualHighlight]   FFmpeg args: ${args.join(' ')}`);

  return new Promise((resolve, reject) => {
    const child = spawn(ffmpeg, args, { windowsHide: true });
    let stderrOutput = '';

    child.stderr.on('data', (data) => {
      stderrOutput += data.toString();
    });

    child.on('error', reject);

    child.on('close', (code) => {
      // Use helper function to parse decoder from output
      const decoderUsed = parseDecoderFromOutput(stderrOutput);
      const hwStatus = getDecoderType(decoderUsed);
      const statusStr = decoderUsed === 'unknown' ? 'unknown' : (hwStatus === 'GPU' ? 'GPU' : 'CPU');

      if (decoderUsed === 'unknown') {
        const snippet = stderrOutput.substring(0, 1500).replace(/\n/g, ' | ');
        logger.log(`[VisualHighlight]   Decoder: unknown - Full FFmpeg output: ${snippet}`);
      }
      logger.log(`[VisualHighlight]   Decoder used: ${decoderUsed} (${statusStr})`);

      if (code === 0 && fs.existsSync(outputPath)) {
        resolve(outputPath);
      } else {
        reject(new Error(stderrOutput || `FFmpeg exited with code ${code}`));
      }
    });
  });
}

async function loadImageRawRgba(imagePath) {
  const sharp = require('sharp');
  return await sharp(imagePath).ensureAlpha().resize(RESIZE_WIDTH, RESIZE_HEIGHT).raw().toBuffer();
}

function computeDirectSimilarityFromBuf(buf1, buf2) {
  const minLen = Math.min(buf1.length, buf2.length);
  let diff = 0;
  for (let i = 0; i < minLen; i += 4) {
    diff += Math.abs(buf1[i] - buf2[i]);
    diff += Math.abs(buf1[i + 1] - buf2[i + 1]);
    diff += Math.abs(buf1[i + 2] - buf2[i + 2]);
  }
  const maxDiff = (minLen / 4) * 255 * 3;
  return maxDiff === 0 ? 1 : 1 - (diff / maxDiff);
}

async function computeNormalizedCorrelation(path1, path2) {
  const buf1 = await loadImageRawRgba(path1);
  const buf2 = await loadImageRawRgba(path2);
  const width = RESIZE_WIDTH;
  const height = RESIZE_HEIGHT;

  const pixels1 = extractPixelsFromRGBA(buf1, width, height);
  const pixels2 = extractPixelsFromRGBA(buf2, width, height);

  if (pixels1.length === 0 || pixels2.length === 0) {
    const sim = computeDirectSimilarityFromBuf(buf1, buf2);
    return sim * 2 - 1; // map 0-1 to -1..1 for (nc+1)/2 in caller
  }
  
  let sum1 = 0, sum2 = 0, sum1Sq = 0, sum2Sq = 0, pSum = 0;
  
  for (let i = 0; i < Math.min(pixels1.length, pixels2.length); i++) {
    const v1 = pixels1[i].r + pixels1[i].g + pixels1[i].b;
    const v2 = pixels2[i].r + pixels2[i].g + pixels2[i].b;
    
    sum1 += v1;
    sum2 += v2;
    sum1Sq += v1 * v1;
    sum2Sq += v2 * v2;
    pSum += v1 * v2;
  }
  
  const numPixels = Math.min(pixels1.length, pixels2.length);
  const num = numPixels * pSum - sum1 * sum2;
  const den = Math.sqrt((numPixels * sum1Sq - sum1 * sum1) * (numPixels * sum2Sq - sum2 * sum2));
  
  if (den === 0) return 0;
  return num / den;
}

function extractPixelsFromRGBA(buffer, width, height) {
  const pixels = [];
  
  for (let i = 0; i < width * height; i++) {
    const offset = i * 4;
    if (offset + 3 < buffer.length) {
      pixels.push({
        r: buffer[offset] || 0,
        g: buffer[offset + 1] || 0,
        b: buffer[offset + 2] || 0,
        a: buffer[offset + 3] || 255
      });
    }
  }
  
  return pixels;
}

async function compareWithMSE(templatePath, framePath, options = {}) {
  try {
    const nc = await computeNormalizedCorrelation(templatePath, framePath);
    const similarity = Math.max(0, Math.min(1, (nc + 1) / 2));
    return { similarity, method: 'mse', nc };
  } catch (error) {
    logger.error('[VisualHighlight] MSE comparison error:', error);
    return { similarity: 0, error: error.message };
  }
}

async function compareWithPixelmatch(templatePath, framePath, options = {}) {
  try {
    const pixelmatch = require('pixelmatch');
    const buf1 = await loadImageRawRgba(templatePath);
    const buf2 = await loadImageRawRgba(framePath);
    const width = RESIZE_WIDTH;
    const height = RESIZE_HEIGHT;
    const numPixels = width * height;
    const diff = Buffer.alloc(numPixels * 4);
    const numDiff = pixelmatch(buf1, buf2, diff, width, height, { threshold: 0.1, alpha: 0.1 });
    const similarity = Math.max(0, 1 - (numDiff / numPixels));
    return { similarity, numDiff };
  } catch (error) {
    logger.error('[VisualHighlight] pixelmatch error:', error);
    return { similarity: 0, numDiff: RESIZE_WIDTH * RESIZE_HEIGHT };
  }
}

async function compareWithOpenCV(templatePath, framePath, options = {}) {
  try {
    const buf1 = await loadImageRawRgba(templatePath);
    const buf2 = await loadImageRawRgba(framePath);
    const similarity = computeDirectSimilarityFromBuf(buf1, buf2);
    return { similarity, method: 'opencv' };
  } catch (error) {
    logger.error('[VisualHighlight] OpenCV comparison error:', error);
    return { similarity: 0, error: error.message };
  }
}

// pHash comparison - perceptual hashing (manual implementation)
async function compareWithPHash(templatePath, framePath, options = {}) {
  try {
    const sharp = require('sharp');
    
    // Resize to 32x32 and convert to grayscale
    const templateBuf = await sharp(templatePath).resize(32, 32).grayscale().raw().toBuffer();
    const frameBuf = await sharp(framePath).resize(32, 32).grayscale().raw().toBuffer();
    
    // Compute simple hash: average pixel value comparison
    const templateHash = computeSimpleHash(templateBuf);
    const frameHash = computeSimpleHash(frameBuf);
    
    // Compare hashes using Hamming distance
    let distance = 0;
    for (let i = 0; i < templateHash.length; i++) {
      if (templateHash[i] !== frameHash[i]) distance++;
    }
    
    // Convert to similarity (0-1)
    const similarity = Math.max(0, 1 - (distance / 64));
    
    return { similarity, method: 'phash', distance };
  } catch (error) {
    logger.error('[VisualHighlight] pHash error:', error);
    return { similarity: 0, error: error.message };
  }
}

function computeSimpleHash(buffer) {
  // Compute 64-bit hash from 32x32 grayscale image
  const size = 32;
  const hash = [];
  
  // Calculate average
  let sum = 0;
  for (let i = 0; i < buffer.length; i++) {
    sum += buffer[i];
  }
  const avg = sum / buffer.length;
  
  // Create binary hash based on average
  for (let i = 0; i < size * size; i++) {
    hash.push(buffer[i] >= avg ? 1 : 0);
  }
  
  return hash;
}

// SSIM comparison - structural similarity
async function compareWithSSIM(templatePath, framePath, options = {}) {
  try {
    const ssim = require('ssim.js').default;
    const sharp = require('sharp');
    
    const templateImg = await sharp(templatePath).resize(64, 64).raw().toBuffer({ resolveWithObject: true });
    const frameImg = await sharp(framePath).resize(64, 64).raw().toBuffer({ resolveWithObject: true });
    
    const templateData = new Uint8ClampedArray(templateImg.data);
    const frameData = new Uint8ClampedArray(frameImg.data);
    
    // Create ImageData-like objects for ssim.js
    const templateImageData = {
      data: templateData,
      width: 64,
      height: 64
    };
    const frameImageData = {
      data: frameData,
      width: 64,
      height: 64
    };
    
    const result = ssim(templateImageData, frameImageData, { 
      k1: 0.01, 
      k2: 0.03,
      bitDepth: 8 
    });
    
    return { similarity: result.mssim, method: 'ssim' };
  } catch (error) {
    logger.error('[VisualHighlight] SSIM error:', error);
    return { similarity: 0, error: error.message };
  }
}

// Multi-scale + Histogram comparison
async function compareWithMultiScale(templatePath, framePath, options = {}) {
  try {
    const sharp = require('sharp');
    const scales = [16, 32, 64];
    let totalScore = 0;
    
    for (const size of scales) {
      const templateBuf = await sharp(templatePath).resize(size, size).raw().toBuffer();
      const frameBuf = await sharp(framePath).resize(size, size).raw().toBuffer();
      
      // Pixel similarity at this scale
      let diff = 0;
      const minLen = Math.min(templateBuf.length, frameBuf.length);
      for (let i = 0; i < minLen; i++) {
        diff += Math.abs(templateBuf[i] - frameBuf[i]);
      }
      const pixelScore = 1 - (diff / (minLen * 255));
      totalScore += pixelScore;
    }
    
    // Histogram comparison
    const templateHist = await getColorHistogram(templatePath);
    const frameHist = await getColorHistogram(framePath);
    const histScore = compareHistograms(templateHist, frameHist);
    
    // Combine: 60% pixel similarity, 40% histogram
    const finalScore = (totalScore / scales.length) * 0.6 + histScore * 0.4;
    
    return { similarity: Math.max(0, Math.min(1, finalScore)), method: 'multiscale' };
  } catch (error) {
    logger.error('[VisualHighlight] MultiScale error:', error);
    return { similarity: 0, error: error.message };
  }
}

async function getColorHistogram(imagePath) {
  const sharp = require('sharp');
  const { data, info } = await sharp(imagePath)
    .resize(32, 32)
    .raw()
    .toBuffer({ resolveWithObject: true });
  
  const hist = { r: new Array(256).fill(0), g: new Array(256).fill(0), b: new Array(256).fill(0) };
  
  for (let i = 0; i < data.length; i += info.channels) {
    hist.r[data[i]]++;
    hist.g[data[i + 1]]++;
    hist.b[data[i + 2]]++;
  }
  
  return hist;
}

function compareHistograms(h1, h2) {
  // Compute histogram intersection
  let intersection = 0;
  let total1 = 0;
  let total2 = 0;
  
  for (let i = 0; i < 256; i++) {
    intersection += Math.min(h1.r[i], h2.r[i]) + Math.min(h1.g[i], h2.g[i]) + Math.min(h1.b[i], h2.b[i]);
    total1 += h1.r[i] + h1.g[i] + h1.b[i];
    total2 += h2.r[i] + h2.g[i] + h2.b[i];
  }
  
  return intersection / Math.max(total1, total2);
}

async function compareRegions(templatePath, framePath, method, options = {}) {
  switch (method) {
    case 'fast':
      return await compareWithMSE(templatePath, framePath, options);
    case 'balanced':
      return await compareWithPixelmatch(templatePath, framePath, options);
    case 'accurate':
      return await compareWithOpenCV(templatePath, framePath, options);
    case 'phash':
      return await compareWithPHash(templatePath, framePath, options);
    case 'ssim':
      return await compareWithSSIM(templatePath, framePath, options);
    case 'multiscale':
      return await compareWithMultiScale(templatePath, framePath, options);
    case 'mobilenet':
      return await gpuModule.compareWithMobileNet(templatePath, framePath);
    default:
      return await compareWithMSE(templatePath, framePath, options);
  }
}

async function extractTemplateFromRegion(videoPath, timestamp, region) {
  const tempDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'miba-visual-'));
  const outputPath = path.join(tempDir, 'template.png');
  
  await extractRegionAsImage(videoPath, timestamp, region, outputPath);
  
  const imageData = fs.readFileSync(outputPath);
  const base64 = imageData.toString('base64');
  
  await fs.promises.rm(tempDir, { recursive: true, force: true }).catch(() => {});
  
  return {
    imageData: base64,
    region,
    timestamp,
    width: RESIZE_WIDTH,
    height: RESIZE_HEIGHT
  };
}

// Segment-based batch extraction: N FFmpeg processes (one per segment) instead of thousands.
// Each process decodes a contiguous segment with GPU, outputs PNG (Sharp requires PNG/JPEG, not BMP).
// Avoids process-spawn bottleneck and keeps GPU decode pipeline saturated.
const EXTRACT_SEGMENTS = Math.min(8, Math.max(4, (typeof os.cpus === 'function' ? os.cpus().length : 0) || 4));

async function extractAllFramesToCache(videoPath, timestamps, region, tempDir, metadata, onProgress) {
  if (timestamps.length === 0) {
    if (onProgress) onProgress(45, 'Extracted 0 frames');
    return [];
  }

  const codec = await getVideoCodec(videoPath);
  const hwDecoder = await getHardwareDecoder(codec);

  logger.log(`[VisualHighlight] extractAllFramesToCache: ${path.basename(videoPath)}`);
  logger.log(`[VisualHighlight]   Total timestamps: ${timestamps.length}, Codec: ${codec || 'unknown'}, Duration: ${metadata.duration}s, FPS: ${metadata.fps}`);

  const scaledRegion = scaleRegionToNative(region, metadata.width, metadata.height);
  const { x, y, width: rw, height: rh } = scaledRegion;
  const frameSkip = Math.max(1, Math.round((timestamps[1] - timestamps[0]) * metadata.fps));
  const fpsFilter = metadata.fps / frameSkip; // e.g. 30/5 = 6
  const duration = metadata.duration;
  const numSegments = Math.min(EXTRACT_SEGMENTS, Math.ceil(duration / 10)); // min 10s per segment
  const segmentDuration = duration / numSegments;
  const filter = `crop=${rw}:${rh}:${x}:${y},scale=${RESIZE_WIDTH}:${RESIZE_HEIGHT}:flags=neighbor,fps=${fpsFilter}`;

  if (hwDecoder) {
    logger.log(`[VisualHighlight]   Segments: ${numSegments}, Segment duration: ${segmentDuration.toFixed(1)}s, Using GPU decoder: ${hwDecoder.decoder} (${hwDecoder.hwaccel})`);
  } else {
    logger.log(`[VisualHighlight]   Segments: ${numSegments}, Segment duration: ${segmentDuration.toFixed(1)}s, HW accel: auto (using software decode)`);
  }

  const { spawn } = require('child_process');
  const ffmpeg = findFFmpegSync();

  function runSegment(segIdx) {
    return new Promise((resolve, reject) => {
      const start = segIdx * segmentDuration;
      const segDuration = segIdx === numSegments - 1 ? duration - start : segmentDuration;
      const segDir = path.join(tempDir, `seg_${segIdx}`);
      fs.mkdirSync(segDir, { recursive: true });
      const outPattern = path.join(segDir, 'frame_%04d.png');

      // Build args with explicit hardware decoder if available
      const args = [];

      if (hwDecoder) {
        args.push('-hwaccel', hwDecoder.hwaccel);
      } else {
        args.push('-hwaccel', 'auto');
      }

      args.push('-y', '-hide_banner', '-loglevel', 'info');
      args.push('-ss', String(start), '-i', videoPath, '-t', String(segDuration));

      // Explicitly specify the decoder (needed for hardware decode)
      if (hwDecoder) {
        args.push('-c:v', hwDecoder.decoder);
      }

      args.push('-vf', filter, '-f', 'image2', '-c:v', 'png', '-compression_level', '0', outPattern);

      const child = spawn(ffmpeg, args, { windowsHide: true });
      let stderr = '';
      child.stderr.on('data', (d) => { stderr += d.toString(); });
      child.on('error', reject);
      child.on('close', (code) => {
        // Use helper function to parse decoder from output
        const decoderUsed = parseDecoderFromOutput(stderr);
        const hwStatus = getDecoderType(decoderUsed);
        const statusStr = decoderUsed === 'unknown' ? 'unknown' : (hwStatus === 'GPU' ? 'GPU' : 'CPU');

        if (segIdx === 0) {
          if (decoderUsed === 'unknown') {
            const snippet = stderr.substring(0, 1500).replace(/\n/g, ' | ');
            logger.log(`[VisualHighlight]   Decoder: unknown - Full FFmpeg output: ${snippet}`);
          }
          logger.log(`[VisualHighlight]   Decoder used: ${decoderUsed} (${statusStr})`);
        }

        if (code !== 0) {
          reject(new Error(stderr || `FFmpeg exit ${code}`));
          return;
        }
        const framePaths = [];
        const expectedFrames = Math.ceil(segDuration * fpsFilter);
        for (let k = 1; k <= expectedFrames; k++) {
          const ts = start + (k - 1) / fpsFilter;
          const fp = path.join(segDir, `frame_${String(k).padStart(4, '0')}.png`);
          if (fs.existsSync(fp)) framePaths.push({ timestamp: ts, path: fp });
        }
        resolve(framePaths);
      });
    });
  }

  const segments = Array.from({ length: numSegments }, (_, i) => i);
  let completedSegments = 0;
  const results = await Promise.all(segments.map((segIdx) => {
    return runSegment(segIdx)
      .then((r) => {
        completedSegments++;
        if (onProgress) onProgress(5 + Math.round((completedSegments / numSegments) * 40), `Extracting: segment ${completedSegments}/${numSegments}`);
        return r;
      })
      .catch((err) => {
        logger.warn(`[VisualHighlight] Segment ${segIdx} failed: ${err.message}, falling back to per-frame`);
        return extractSegmentFallback(videoPath, timestamps, region, tempDir, segIdx, numSegments, duration, metadata);
      });
  }));

  const framePaths = results.flat().sort((a, b) => a.timestamp - b.timestamp);

  if (onProgress) onProgress(45, `Extracted ${framePaths.length} frames`);
  return framePaths;
}

async function extractSegmentFallback(videoPath, timestamps, region, tempDir, segIdx, numSegments, duration, metadata) {
  const segmentDuration = duration / numSegments;
  const start = segIdx * segmentDuration;
  const end = segIdx === numSegments - 1 ? duration : start + segmentDuration;
  const inSegment = timestamps.filter((t) => t >= start && t < end);
  if (inSegment.length === 0) return [];

  const segDir = path.join(tempDir, `seg_${segIdx}`);
  fs.mkdirSync(segDir, { recursive: true });
  const concurrency = 4;
  let idx = 0;

  async function worker() {
    const out = [];
    while (true) {
      const i = idx++;
      if (i >= inSegment.length) return out;
      const ts = inSegment[i];
      const fp = path.join(segDir, `frame_${String(i + 1).padStart(4, '0')}.png`);
      try {
        await extractRegionAsImage(videoPath, ts, region, fp, metadata?.width, metadata?.height);
        out.push({ timestamp: ts, path: fp });
      } catch (e) {
        logger.warn(`[VisualHighlight] Frame at ${ts}s failed:`, e.message);
      }
    }
  }

  const workers = Array.from({ length: concurrency }, () => worker());
  return (await Promise.all(workers)).flat();
}

async function processFramesInBatches(framePaths, templatePath, method, threshold, batchSize, minGapSeconds, onProgress) {
  const rawMatches = [];
  const frameResults = [];

  // Determine number of workers based on CPU cores
  const numWorkers = Math.min(os.cpus().length, 8);
  const workers = [];
  const workerPath = path.join(__dirname, 'visualHighlightWorker.js');

  logger.log(`[VisualHighlight] Starting ${numWorkers} worker threads for frame processing`);

  // Create worker pool
  for (let i = 0; i < numWorkers; i++) {
    const worker = new Worker(workerPath);
    workers.push({ worker, busy: false, results: [] });
  }

  // Function to process a batch using workers
  async function processWithWorkers(batch) {
    return new Promise((resolve) => {
      const results = [];
      let pending = batch.length;
      let workerIndex = 0;

      // Distribute work across workers
      const chunkSize = Math.ceil(batch.length / workers.length);
      let workerPromises = [];

      for (let w = 0; w < workers.length; w++) {
        const start = w * chunkSize;
        const end = Math.min(start + chunkSize, batch.length);
        const chunk = batch.slice(start, end);

        if (chunk.length === 0) continue;

        const workerInfo = workers[w];

        workerPromises.push(new Promise((res) => {
          workerInfo.worker.once('message', (response) => {
            results.push(...response.results);
            res();
          });
          workerInfo.worker.postMessage({
            templatePath,
            framePaths: chunk,
            method,
            threshold
          });
        }));
      }

      Promise.all(workerPromises).then(() => {
        resolve(results);
      });
    });
  }

  // Process frames in batches
  for (let i = 0; i < framePaths.length; i += batchSize) {
    const batch = framePaths.slice(i, i + batchSize);

    if (onProgress) {
      onProgress(50 + Math.round((i / framePaths.length) * 45), `Processing frames: ${Math.min(i + batchSize, framePaths.length)}/${framePaths.length}`);
    }

    const batchResults = await processWithWorkers(batch);

    for (const item of batchResults) {
      if (!item) continue;
      const { frame, result, thumbnailBase64 } = item;
      frameResults.push({
        timestamp: frame.timestamp,
        similarity: result.similarity,
        method: result.method,
        thumbnail: thumbnailBase64
      });
      if (result.similarity >= threshold) {
        rawMatches.push({
          startTime: frame.timestamp,
          similarity: result.similarity,
          method,
          thumbnail: thumbnailBase64
        });
      }
    }
  }

  // Sort all above-threshold matches by timestamp, then apply min-gap filter
  // to guarantee the first occurrence in each cluster is always selected
  rawMatches.sort((a, b) => a.startTime - b.startTime);
  const matches = [];
  let lastMatchEnd = -minGapSeconds;
  for (const m of rawMatches) {
    if (m.startTime - lastMatchEnd >= minGapSeconds) {
      matches.push(m);
      lastMatchEnd = m.startTime;
    }
  }

  // Clean up workers
  for (const w of workers) {
    w.worker.terminate();
  }

  logger.log(`[VisualHighlight] Worker threads completed processing ${frameResults.length} frames`);

  return { matches, frameResults };
}

async function detectVisualHighlights({
  videoPath,
  templateBase64,
  region,
  method = 'fast',
  threshold = DEFAULT_SIMILARITY_THRESHOLD,
  frameSkip = DEFAULT_FRAME_SKIP,
  batchSize = DEFAULT_BATCH_SIZE,
  minGapSeconds = 2,
  onProgress
}) {
  const report = (p, msg) => {
    if (typeof onProgress === 'function') onProgress(p, msg);
    logger.log(`[VisualHighlight] ${msg} (${Math.round(p)}%)`);
  };

  // Log the method being used
  logger.log(`[VisualHighlight] Processing method: ${method}`);

  // Handle 'gpu' preset - maps to mobilenet for GPU-accelerated processing
  let actualMethod = method;
  if (method === 'gpu') {
    report(1, 'Initializing GPU-accelerated processing...');
    try {
      await gpuModule.initialize();
      const gpuStatus = await gpuModule.checkGpuAvailability();
      report(2, `GPU status: ${gpuStatus.backend} (${gpuStatus.available ? 'accelerated' : 'CPU fallback'})`);
      actualMethod = 'mobilenet';
      logger.log(`[VisualHighlight] GPU processing initialized with backend: ${gpuStatus.backend}`);
    } catch (gpuError) {
      logger.error('[VisualHighlight] GPU initialization failed:', gpuError.message);
      report(2, `GPU failed: ${gpuError.message}, using CPU`);
      actualMethod = 'fast'; // Fallback to CPU
    }
  }

  logger.log(`[VisualHighlight] Using comparison method: ${actualMethod}`);

  try {
    if (!videoPath || !fs.existsSync(videoPath)) {
      return { success: false, error: 'Video file not found.' };
    }

    if (!templateBase64) {
      return { success: false, error: 'Template image is required.' };
    }

    const tempDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'miba-visual-'));
    
    const templatePath = path.join(tempDir, 'template.png');
    fs.writeFileSync(templatePath, Buffer.from(templateBase64, 'base64'));
    
    report(2, 'Getting video metadata...');
    
    const metadata = await getVideoMetadata(videoPath);
    
    if (!metadata.duration || metadata.duration <= 0) {
      await fs.promises.rm(tempDir, { recursive: true, force: true }).catch(() => {});
      return { success: false, error: 'Could not determine video duration.' };
    }
    
    const totalFrames = Math.floor(metadata.duration * metadata.fps);
    const frameIndices = [];
    for (let i = 0; i < totalFrames; i += frameSkip) {
      frameIndices.push(i);
    }
    
    const timestamps = frameIndices.map(i => i / metadata.fps);
    
    logger.log(`[VisualHighlight] Video: ${metadata.duration.toFixed(1)}s, ${metadata.fps}fps, ${totalFrames} frames. Scanning ${timestamps.length} frames (every ${frameSkip}th), method: ${actualMethod}, batch: ${batchSize}`);
    
    report(5, `Extracting ${timestamps.length} frames...`);
    
    const framePaths = await extractAllFramesToCache(videoPath, timestamps, region, tempDir, metadata, report);
    
    report(50, `Processing ${framePaths.length} frames in batches of ${batchSize}...`);
    
    const { matches, frameResults } = await processFramesInBatches(
      framePaths,
      templatePath,
      actualMethod,
      threshold,
      batchSize,
      minGapSeconds,
      report
    );
    
    await fs.promises.rm(tempDir, { recursive: true, force: true }).catch(() => {});
    
    report(100, `Found ${matches.length} match(es).`);
    
    const cuePointMatches = matches.map((m) => ({
      startTime: m.startTime,
      endTime: m.startTime + 2,
      title: 'Visual Highlight',
      source: 'visual-scan',
      similarity: m.similarity,
      method: m.method
    }));
    
    return { 
      success: true, 
      matches: cuePointMatches,
      frameResults: frameResults.map(r => ({
        timestamp: r.timestamp,
        similarity: r.similarity,
        thumbnail: r.thumbnail
      }))
    };
    
  } catch (error) {
    logger.error('[VisualHighlight] Error:', error);
    return {
      success: false,
      error: error.message || 'Visual highlight detection failed.'
    };
  }
}

module.exports = {
  extractTemplateFromRegion,
  detectVisualHighlights,
  compareRegions,
  RESIZE_WIDTH,
  RESIZE_HEIGHT,
  DEFAULT_FRAME_SKIP,
  DEFAULT_BATCH_SIZE,
  DEFAULT_SIMILARITY_THRESHOLD
};
